package com.flowingcode.vaadin.addons.gridhelpers;

import java.io.Serializable;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.vaadin.flow.component.AttachEvent;
import com.vaadin.flow.component.ComponentUtil;
import com.vaadin.flow.component.checkbox.Checkbox;
import com.vaadin.flow.component.contextmenu.MenuItem;
import com.vaadin.flow.component.contextmenu.SubMenu;
import com.vaadin.flow.component.dependency.CssImport;
import com.vaadin.flow.component.dependency.JsModule;
import com.vaadin.flow.component.grid.Grid;
import com.vaadin.flow.component.grid.Grid.Column;
import com.vaadin.flow.component.grid.Grid.SelectionMode;
import com.vaadin.flow.component.grid.GridMultiSelectionModel;
import com.vaadin.flow.component.grid.GridSelectionModel;
import com.vaadin.flow.component.grid.GridSingleSelectionModel;
import com.vaadin.flow.component.grid.ItemClickEvent;
import com.vaadin.flow.component.icon.VaadinIcon;
import com.vaadin.flow.component.menubar.MenuBar;
import com.vaadin.flow.component.menubar.MenuBarVariant;
import com.vaadin.flow.function.SerializableFunction;
import com.vaadin.flow.function.SerializablePredicate;

import lombok.Getter;
import lombok.NonNull;
import lombok.Setter;

@SuppressWarnings("serial")
@Getter
@JsModule("./fcGridHelper/connector.js")
@CssImport(value = "./fcGridHelper/vaadin-menu-bar.css", themeFor = "vaadin-menu-bar")
@CssImport(value = GridHelper.GRID_STYLES, themeFor = "vaadin-grid")
public final class GridHelper<T> implements Serializable {

  private static final String GRID_HELPER_TOGGLE_THEME = "gridHelperToggle";

  private final static String TOGGLE_LABEL_DATA = GridHelper.class.getName() + "#TOGGLE_LABEL";
  
  public final static  String GRID_STYLES = "./fcGridHelper/vaadin-grid.css";

  /**Compact row styling for Vaadin Grid*/
  // https://cookbook.vaadin.com/grid-dense-theme
  public final static String DENSE_THEME = "fcGh-dense";

  @Getter
  private final Grid<T> grid;
  private final GridHelperClassNameGenerator<T> helperClassNameGenerator;

  private boolean selectionColumnHidden;
  private boolean selectionColumnFrozen;
  private boolean arrowSelectionEnabled;
  private Column<T> menuToggleColumn;


  @Setter
  private boolean selectOnClick;

  private SerializablePredicate<T> selectionFilter;

  private GridHelper(Grid<T> grid) {
    this.grid = grid;
    this.helperClassNameGenerator = new GridHelperClassNameGenerator<>();
    grid.addItemClickListener(this::onItemClick);
    grid.addAttachListener(this::onAttach);
  }

  public static <T> GridHelper<T> extend(Grid<T> grid) {
    @SuppressWarnings("unchecked")
    GridHelper<T> helper = ComponentUtil.getData(grid, GridHelper.class);
    if (helper == null) {
      helper = new GridHelper<>(grid);
      ComponentUtil.setData(grid, GridHelper.class, helper);
    }
    return helper;
  }

  private void initConnector() {
    grid.getUI().orElseThrow(
        () -> new IllegalStateException("Connector can only be initialized for an attached Grid"))
        .getPage()
        .executeJs("window.Vaadin.Flow.fcGridHelperConnector.initLazy($0)", grid.getElement());
  }

  private void onAttach(AttachEvent event) {
    initConnector();
    if (isSelectionColumnFrozen()) {
      setSelectionColumnFrozen(true);
    }

    if (isSelectionColumnHidden()) {
      setSelectionColumnHidden(true);
    }
  }

  private void onItemClick(ItemClickEvent<T> event) {
    T item = event.getItem();
    if (selectOnClick) {
      // https://cookbook.vaadin.com/grid-conditional-select
      if (!canSelect(item)) {
        return;
      }

      if (grid.getSelectedItems().contains(item)) {
        grid.deselect(item);
      } else {
        grid.select(item);
      }
    }
  }

  private boolean canSelect(T item) {
    return selectionFilter == null || selectionFilter.test(item);
  }

  /** Return the grid selection mode */
  public SelectionMode getSelectionMode() {
    GridSelectionModel<?> model = grid.getSelectionModel();
    if (model instanceof GridSingleSelectionModel) {
      return SelectionMode.SINGLE;
    }
    if (model instanceof GridMultiSelectionModel) {
      return SelectionMode.MULTI;
    }
    return SelectionMode.NONE;
  }

  /**
   * Sets the function that is used for generating CSS class names for all the
   * cells in the rows in this grid. Returning {@code null} from the generator
   * results in no custom class name being set. Multiple class names can be
   * returned from the generator as space-separated.
   * <p>
   * If {@link Column#setClassNameGenerator(SerializableFunction)} is used
   * together with this method, resulting class names from both methods will
   * be effective. Class names generated by grid are applied to the cells
   * before the class names generated by column. This means that if the
   * classes contain conflicting style properties, column's classes will win.
   *
   * @param classNameGenerator
   *            the class name generator to set, not {@code null}
   * @throws NullPointerException
   *             if {@code classNameGenerator} is {@code null}
   * @see Column#setClassNameGenerator(SerializableFunction)
   */
  public void setClassNameGenerator(SerializableFunction<T, String> classNameGenerator) {
    grid.setClassNameGenerator(this.helperClassNameGenerator);
    if (classNameGenerator instanceof GridHelperClassNameGenerator) {
      helperClassNameGenerator.setGridClassNameGenerator(
          ((GridHelperClassNameGenerator<T>) classNameGenerator).getGridClassNameGenerator());
    } else {
      helperClassNameGenerator.setGridClassNameGenerator(classNameGenerator);
    }
  }

  /** Set whether the multiselect selection column is hidden. */
  public void setSelectionColumnHidden(boolean value) {
    //https://cookbook.vaadin.com/grid-multiselect-no-selectcolumn
    selectionColumnHidden = value;
    grid.getElement().executeJs(
        "this.getElementsByTagName('vaadin-grid-flow-selection-column')[0].hidden = $0;", value);
  }

  /** Set whether the multiselect selection column is frozen. */
  public void setSelectionColumnFrozen(boolean value) {
    //https://cookbook.vaadin.com/grid-frozen-selection-column
    selectionColumnFrozen = value;
    grid.getElement()
        .executeJs("this.querySelector('vaadin-grid-flow-selection-column').frozen = $0", value);
  }

  /** Allow Grid rows to be selected using up/down arrow keys */
  public void setArrowSelectionEnabled(boolean value) {
    grid.getElement().setProperty("_fcghArrowSelection", value);
  }

  private void deselectIf(SerializablePredicate<T> predicate) {
    switch (getSelectionMode()) {
      case MULTI:
        grid.asMultiSelect().deselect(grid.asMultiSelect().getSelectedItems().stream()
            .filter(predicate).collect(Collectors.toList()));
        break;
      case SINGLE:
        grid.asSingleSelect().getOptionalValue().filter(predicate)
            .ifPresent(x -> grid.asSingleSelect().clear());
        break;
      default:
        break;
    }
  }

  /** Set a predicate for determining which rows are selectable. */
  public void setSelectionFilter(SerializablePredicate<T> predicate) {
    this.selectionFilter = predicate;
    if (predicate != null) {
      deselectIf(predicate.negate());
    }
  }

  /** Show a menu to toggle the visibility of grid columns */
  public void setColumnToggleVisible(boolean visible) {
    // https://cookbook.vaadin.com/grid-column-toggle
    if (visible) {
      showColumnToggle();
    } else {
      hideColumnToggle();
    }
  }

  public boolean isColumnToggleVisible() {
    return menuToggleColumn != null && menuToggleColumn.isVisible();
  }

  private void showColumnToggle() {
    createMenuToggle().ifPresent(toggle -> {
      if (menuToggleColumn == null) {
        menuToggleColumn = grid.addColumn(t -> "").setWidth("0").setFlexGrow(0);
      } else {
        menuToggleColumn.setVisible(true);
      }
      grid.getHeaderRows().get(0).getCell(menuToggleColumn).setComponent(toggle);
    });
  }

  private void hideColumnToggle() {
    if (menuToggleColumn != null) {
      menuToggleColumn.setVisible(false);
    }
  }

  private Optional<MenuBar> createMenuToggle() {
    MenuBar menuBar = new MenuBar();
    menuBar.addThemeVariants(MenuBarVariant.LUMO_TERTIARY_INLINE);
    MenuItem menuItem = menuBar.addItem(VaadinIcon.ELLIPSIS_DOTS_V.create());
    SubMenu subMenu = menuItem.getSubMenu();

    for (Column<T> column : grid.getColumns()) {
      getToggleLabel(column).ifPresent(label->{
        Checkbox checkbox = new Checkbox(label);
        checkbox.setValue(column.isVisible());
        checkbox.addValueChangeListener(e -> column.setVisible(e.getValue()));
        subMenu.addItem(checkbox);
      });
    }

    menuBar.getThemeNames().add(GRID_HELPER_TOGGLE_THEME);
    return Optional.of(menuBar).filter(_menuBar -> !_menuBar.getItems().isEmpty());
  }

  private Optional<String> getToggleLabel(@NonNull Column<T> column) {
    return Optional.ofNullable((String) ComponentUtil.getData(column, TOGGLE_LABEL_DATA));
  }

  public void setColumnToggleLabel(Column<T> column, String label) {
    if (column != null) {
      if (!grid.getColumns().contains(column)) {
        throw new IllegalArgumentException();
      }
      ComponentUtil.setData(column, TOGGLE_LABEL_DATA, label);
      if (isColumnToggleVisible()) {
        showColumnToggle();
      }
    }
  }

  public List<Column<T>> getColumns() {
    return grid.getColumns().stream().filter(c -> c != menuToggleColumn)
        .collect(Collectors.toList());
  }

  // - Make a responsive Grid that has a different set of columns depending on the browser width
  // https://cookbook.vaadin.com/responsive-grid

  // - Show a meaningful message instead of an empty Grid
  // https://cookbook.vaadin.com/grid-message-when-empty
}
